<?php
declare(strict_types=1);

require_once __DIR__ . '/../db.php';
require_once __DIR__ . '/../helpers.php';

function get_min_total_pesos(): int {
  $cfg = require __DIR__ . '/../config.php';
  $min = (int)$cfg['min_total_pesos'];
  try {
    $row = db()->query("SELECT value FROM settings WHERE key='min_total_pesos'")->fetch();
    if ($row && is_numeric($row['value'])) $min = (int)$row['value'];
  } catch (Throwable $e) {}
  return $min;
}

function export_files_for_period(string $period): array {
  if (!is_valid_period($period)) {
    throw new RuntimeException("Período inválido. Formato esperado YYYYMM.");
  }

  $cfg = require __DIR__ . '/../config.php';
  $exportsDir = $cfg['exports_dir'];
  if (!is_dir($exportsDir)) mkdir($exportsDir, 0775, true);

  $pdo = db();
  $minTotal = get_min_total_pesos();

  // Total por deudor en pesos (sumatoria de importes del período)
  $totals = [];
  $stmt = $pdo->prepare("SELECT debtor_id, SUM(amount_pesos) AS total_pesos
                         FROM debtor_assists
                         WHERE period = :p
                         GROUP BY debtor_id");
  $stmt->execute([':p'=>$period]);
  foreach ($stmt->fetchAll() as $r) {
    $totals[(int)$r['debtor_id']] = (int)$r['total_pesos'];
  }

  // Deudores involucrados en el período, con total >= mínimo
  $debtorIds = array_keys(array_filter($totals, fn($t)=>$t >= $minTotal));

  // PROVEEDORES
  $providersLines = [];
  if (!empty($debtorIds)) {
    $in = implode(',', array_fill(0, count($debtorIds), '?'));
    $q = "SELECT * FROM debtors WHERE id IN ($in) ORDER BY id_number";
    $st = $pdo->prepare($q);
    $st->execute($debtorIds);
    $debtors = $st->fetchAll();

    foreach ($debtors as $d) {
      $id = (int)$d['id'];
      $totalThousands = pesos_to_thousands_int((int)$totals[$id]);
      if ($totalThousands <= 0) continue;

      $idType = (int)$d['id_type'];
      $idNumber = (string)$d['id_number'];

      $name = normalize_name((string)$d['name']);
      $sit = (string)$d['situation_code'];
      $art26 = (int)$d['art26_flag'];
      $recat = (int)$d['recat_flag'];
      $days = $d['days_overdue'] === null ? '' : (string)(int)$d['days_overdue'];

      $sitNoRe = (string)$d['situation_without_reclass'];

      // Si recat=0 => days debe ser nulo y sitNoRe=00
      if ($recat === 0) {
        $days = '';
        $sitNoRe = '00';
      }

      $providersLines[] = implode(';', [
        (string)$idType,
        $idNumber,
        $name,
        $sit,
        (string)$totalThousands,
        (string)$art26,
        (string)$recat,
        $days === '' ? '' : str_pad($days, 4, '0', STR_PAD_LEFT),
        $sitNoRe
      ]);
    }
  }

  // IMPORTES
  $importesLines = [];
  if (!empty($debtorIds)) {
    $in = implode(',', array_fill(0, count($debtorIds), '?'));
    $q = "SELECT da.*, d.id_type, d.id_number
          FROM debtor_assists da
          JOIN debtors d ON d.id = da.debtor_id
          WHERE da.period = ? AND da.debtor_id IN ($in)
          ORDER BY d.id_number, da.assist_code";
    $args = array_merge([$period], $debtorIds);
    $st = $pdo->prepare($q);
    $st->execute($args);
    foreach ($st->fetchAll() as $r) {
      $th = pesos_to_thousands_int((int)$r['amount_pesos']);
      if ($th <= 0) continue;
      $importesLines[] = implode(';', [
        (string)(int)$r['id_type'],
        (string)$r['id_number'],
        (string)$r['assist_code'],
        (string)$th
      ]);
    }
  }

  // TASA
  $rate = $pdo->prepare("SELECT * FROM rates WHERE period = :p");
  $rate->execute([':p'=>$period]);
  $rr = $rate->fetch();
  if (!$rr) {
    // default
    $noReal = 0;
    $w = "000,00";
  } else {
    $noReal = (int)$rr['no_real_guarantee'];
    $w = (string)$rr['weighted_rate'];
  }
  // Validación básica: formato EEE,DD
  if (!preg_match('/^\d{3},\d{2}$/', $w)) {
    throw new RuntimeException("Tasa inválida: debe ser EEE,DD (ej: 125,50).");
  }
  if ($noReal === 0 && $w !== "000,00") {
    throw new RuntimeException("TASA: Si campo 1 = 0, campo 2 debe ser 000,00.");
  }
  if ($noReal === 1 && $w === "000,00") {
    // permitido? La norma dice inconsistencia (campo1=1 y campo2=000,00) => error 301
    // Forzamos a advertir:
    throw new RuntimeException("TASA: Si campo 1 = 1, la tasa no puede ser 000,00 (error 301).");
  }
  $tasaLine = (string)$noReal . ';' . $w;

  // Escribir archivos (CRLF + ANSI-1252)
  $files = [];
  $files[] = write_export_file($exportsDir, "PROVEEDORES_{$period}.TXT", $providersLines, $cfg['export_encoding']);
  $files[] = write_export_file($exportsDir, "IMPORTES_{$period}.TXT", $importesLines, $cfg['export_encoding']);
  $files[] = write_export_file($exportsDir, "TASA_{$period}.TXT", [$tasaLine], $cfg['export_encoding']);

  return $files;
}

function write_export_file(string $dir, string $name, array $lines, string $encoding): array {
  $path = rtrim($dir,'/') . '/' . $name;
  $body = crlf_join($lines);
  $body = export_encode($body, $encoding);
  file_put_contents($path, $body);
  return [
    'name' => $name,
    'path' => $path,
    'bytes' => filesize($path) ?: 0,
  ];
}
